(* These functions are wrappers around the functions exposed by the POSIX interface of the kernel.
   When the system does not provide such function directly, we implement it ourselves.
   Refer to the POSIX specification for semantics, which can be found at http://pubs.opengroup.org/onlinepubs/9699919799/ , volume "System interfaces" .
   
   
   The POSIX interface is not language independent, it is described in the C syntax, which is unfortunate since C impose some constraints on the semantics. For our interface, signatures are given in Bee. We try to stay close to the C signature. In particular, our functions return an integer return code instead of an option, which would be more idiomatic to Bee.
   However, sometimes it makes little sense to mimic C signatures exactly so we take some freedom and do the right thing instead. In particular, sometimes we might want to have return values not as arguments like in C but as return value, which is more idiomatic to Bee. Similarly, we return -errno in the case of an error instead of return -1 and setting errno like in C.
   
   TODO: The type awkwardness of POSIX originates from the lack of a rich type system. Some types are carried at runtime to ensure type safety, like in sockaddr. Therefore, the POSIX interface should be enriched with types to enforce safety at compile time. For example, the types "socket" (an int in POSIX) and "sockaddr" should be parameterized with a socket domain (also called family) to enforce consistency between parameters in the bind() call for instance.
   
*)

module type Posix = 
  type flag = O_CREAT | O_EXCL | O_NOCTTY | O_TRUNC | O_APPEND |
               O_DSYNC | O_NONBLOCK | O_RSYNC | O_SYNC | O_RDONLY | O_RDWR | O_WRONLY | FD_CLOEXEC
  type fcntl_cmd = F_DUPFD | F_DUPFD_CLOEXEC | F_GETFD | F_SETFD | F_GETFL | F_SETFL | F_GETLK |
                   F_SETLK | F_SETLKW | F_GETOWN | F_SETOWN
  type pollflags = POLLIN | POLLOUT | POLLERR | POLLHUP | POLLNVAL
  type errorcode = EAGAIN | EWOULDBLOCK (* TODO: *)
  
  
  type result α = PosixError of int | Result of α (* a general type for the result of a Posix function *)
  
  (* see http://pubs.opengroup.org/onlinepubs/009695399/basedefs/sys/stat.h.html *)
  type stat = {
      st_dev: int;    (* Device ID of device containing file. *)
      st_ino: int;    (* File serial number. *)
      st_mode: int;   (* Mode of file (see below). *)
      st_nlink: int;  (* Number of hard links to the file. *)
      st_uid: int;    (* User ID of file. *)
      st_gid: int;    (* Group ID of file. *)
      st_rdev: int;   (* Device minor number *)
      st_size: int;   (* Size in bytes *)
      st_atime: int;  (* Last access time *)
      st_mtime: int;  (* Last modification time *)
      st_ctime: int;  (* Last status change time *)
      st_blksize: int;
      st_blocks: int; (* Number of blocks allocated for this object. *)
    }
  
  val error_of_int: int -> errorcode (* TODO *)
               
  val read: int -> string -> int -> int -> int  (* file descriptor, buffer, size, offset *)
  val write: int -> string -> int -> int -> int (* file descriptor, buffer, size, offset *)
  val open: string -> (array flag) -> int
  val close: int -> int
  val stat: string -> (result stat)
  val fstat: int -> (result stat)
  val lstat: string -> (result stat)
  val lseek: int -> int -> int -> int
  
  val unlink: string -> int
  
  
  val openat: int -> string -> (array flag) -> int
  
  
  (* sockets *)
  (* type ipv6 
     type ipv4 *)
  type socket_domain = AF_INET | AF_INET6 | AF_UNIX | AF_UNSPEC
  type socket_type = SOCK_STREAM | SOCK_DGRAM | SOCK_SEQPACKET
  type msg_flags = MSG_CTRUNC | MSG_DONTROUTE | MSG_EOR | MSG_OOB | MSG_PEEK | MSG_TRUNC | MSG_WAITALL
  
  type in_addr
  (*type sockaddr α*)
  type sockaddr = ADDR_INET of in_addr * int | ADDR_INET6 of in_addr * int | ADDR_UNIX
  
  val poll: (list (int * (list pollflags))) -> int -> int -> (int * (list int) * (list int) * (list int)) (* TODO: i think we can remove the parens *)
  val select: list int -> list int -> list int -> float -> (int * (list int) * (list int) * (list int)) (* TODO: i think we can remove the parens *)
  val in_addr_of_string: string -> in_addr (* TODO: could be implemented in Bee and return an option instead *)
  val socket: socket_domain -> socket_type -> int -> int
  val connect: int -> sockaddr -> int
  val accept: int -> sockaddr -> int
  val recv: int -> string -> int -> int -> int (* socket, buffer, offset, length *)
  val send: int -> string -> int -> int -> int (* socket, buffer, offset, length *)
  val bind: int -> sockaddr -> int
  val listen: int -> int -> int



  val int_of_flag: flag -> int (* to convert a fcntl flag to the corresponding integer, allows adding flags together, which is common in C *)
  val fcntl: int -> fcntl_cmd -> int -> int
  val dup2: int -> int -> int
  val nanosleep: int -> int -> unit (* TODO: maybe change the signature *)
  val fork: unit -> int
  val execve: string -> (array string) -> (array string) -> int
  val exit: ∀ α . int -> α (* because exit terminates the process, its return value can unify with anything *)
  val clock_gettime_monotonic: unit -> (int64 * int)   (* returns the number of seconds and the number of nanoseconds - not that the value of nanoseconds is necessarily meaningful, on a system with a tick of 250Hz, it is not *)
  val clock_gettime_realtime: unit -> (int64 * int)   (* same *)
  
  val system: string -> int
  
  (* process uid - gid *)
  val getuid: unit -> int
  val setuid: int -> int
  val setgid: int -> int
  
  (* file systems *)
  val chdir: string -> int
  val mkdir: string -> int -> int
  val rmdir: string -> int
  type mount_flags = MS_RDONLY | MS_NOSUID | MS_NODEV | MS_NOEXEC | MS_SYNCHRONOUS | MS_REMOUNT | MS_MANDLOCK | MS_DIRSYNC | MS_NOATIME | MS_NODIRATIME | MS_BIND | MS_MOVE | MS_REC | MS_SILENT | MS_POSIXACL | MS_UNBINDABLE | MS_PRIVATE | MS_SLAVE | MS_SHARED | MS_RELATIME | MS_KERNMOUNT | MS_I_VERSION | MS_STRICTATIME | MS_LAZYTIME | MS_NOSEC | MS_BORN | MS_ACTIVE | MS_NOUSER
  val chroot: string -> int
  val mount: string -> string -> string -> (array mount_flags) -> int
  val umount: string -> int -> int
  val swapon: string -> int -> int
  val readdir: int -> (int * (array string))
  
  (* process environment *)
  val setenv: string -> string -> int -> int
  val getenv: string -> string
  val getenvironment: unit -> (array string)
endmodule
